/*
 * Copyright (C) 2020, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef SAFEPRIMES_H
#define SAFEPRIMES_H

#include <stdlib.h>

#include "binhexbin.h"
#include "parser.h"

#ifdef __cplusplus
extern "C"
{
#endif

struct safeprimes {
	char *p;
	char *q;
	char *g;
	struct buffer p_b;
	struct buffer q_b;
	struct buffer g_b;
};

/* q = (p - 1) / 2 */
static struct safeprimes safeprimes[] = { {
	/* rfc3526 group 14 "2048-bit MODP Group" */
	.p = (char *)
	     "FFFFFFFF" "FFFFFFFF" "C90FDAA2" "2168C234" "C4C6628B" "80DC1CD1"
	     "29024E08" "8A67CC74" "020BBEA6" "3B139B22" "514A0879" "8E3404DD"
	     "EF9519B3" "CD3A431B" "302B0A6D" "F25F1437" "4FE1356D" "6D51C245"
	     "E485B576" "625E7EC6" "F44C42E9" "A637ED6B" "0BFF5CB6" "F406B7ED"
	     "EE386BFB" "5A899FA5" "AE9F2411" "7C4B1FE6" "49286651" "ECE45B3D"
	     "C2007CB8" "A163BF05" "98DA4836" "1C55D39A" "69163FA8" "FD24CF5F"
	     "83655D23" "DCA3AD96" "1C62F356" "208552BB" "9ED52907" "7096966D"
	     "670C354E" "4ABC9804" "F1746C08" "CA18217C" "32905E46" "2E36CE3B"
	     "E39E772C" "180E8603" "9B2783A2" "EC07A28F" "B5C55DF0" "6F4C52C9"
	     "DE2BCBF6" "95581718" "3995497C" "EA956AE5" "15D22618" "98FA0510"
	     "15728E5A" "8AACAA68" "FFFFFFFF" "FFFFFFFF",
	.q = (char *)
	     "7fffffff" "ffffffff" "e487ed51" "10b4611a" "62633145" "c06e0e68"
	     "94812704" "4533e63a" "0105df53" "1d89cd91" "28a5043c" "c71a026e"
	     "f7ca8cd9" "e69d218d" "98158536" "f92f8a1b" "a7f09ab6" "b6a8e122"
	     "f242dabb" "312f3f63" "7a262174" "d31bf6b5" "85ffae5b" "7a035bf6"
	     "f71c35fd" "ad44cfd2" "d74f9208" "be258ff3" "24943328" "f6722d9e"
	     "e1003e5c" "50b1df82" "cc6d241b" "0e2ae9cd" "348b1fd4" "7e9267af"
	     "c1b2ae91" "ee51d6cb" "0e3179ab" "1042a95d" "cf6a9483" "b84b4b36"
	     "b3861aa7" "255e4c02" "78ba3604" "650c10be" "19482f23" "171b671d"
	     "f1cf3b96" "0c074301" "cd93c1d1" "7603d147" "dae2aef8" "37a62964"
	     "ef15e5fb" "4aac0b8c" "1ccaa4be" "754ab572" "8ae9130c" "4c7d0288"
	     "0ab9472d" "45565534" "7fffffff" "ffffffff",
	.g = (char *) "2",
	.p_b = { NULL, 0, },
	.q_b = { NULL, 0, },
	.g_b = { NULL, 0, },
}, {
	/* rfc3526 group 15 "3072-bit MODP Group" */
	.p = (char *)
	     "FFFFFFFF" "FFFFFFFF" "C90FDAA2" "2168C234" "C4C6628B" "80DC1CD1"
	     "29024E08" "8A67CC74" "020BBEA6" "3B139B22" "514A0879" "8E3404DD"
	     "EF9519B3" "CD3A431B" "302B0A6D" "F25F1437" "4FE1356D" "6D51C245"
	     "E485B576" "625E7EC6" "F44C42E9" "A637ED6B" "0BFF5CB6" "F406B7ED"
	     "EE386BFB" "5A899FA5" "AE9F2411" "7C4B1FE6" "49286651" "ECE45B3D"
	     "C2007CB8" "A163BF05" "98DA4836" "1C55D39A" "69163FA8" "FD24CF5F"
	     "83655D23" "DCA3AD96" "1C62F356" "208552BB" "9ED52907" "7096966D"
	     "670C354E" "4ABC9804" "F1746C08" "CA18217C" "32905E46" "2E36CE3B"
	     "E39E772C" "180E8603" "9B2783A2" "EC07A28F" "B5C55DF0" "6F4C52C9"
	     "DE2BCBF6" "95581718" "3995497C" "EA956AE5" "15D22618" "98FA0510"
	     "15728E5A" "8AAAC42D" "AD33170D" "04507A33" "A85521AB" "DF1CBA64"
	     "ECFB8504" "58DBEF0A" "8AEA7157" "5D060C7D" "B3970F85" "A6E1E4C7"
	     "ABF5AE8C" "DB0933D7" "1E8C94E0" "4A25619D" "CEE3D226" "1AD2EE6B"
	     "F12FFA06" "D98A0864" "D8760273" "3EC86A64" "521F2B18" "177B200C"
	     "BBE11757" "7A615D6C" "770988C0" "BAD946E2" "08E24FA0" "74E5AB31"
	     "43DB5BFC" "E0FD108E" "4B82D120" "A93AD2CA" "FFFFFFFF" "FFFFFFFF",
	.q = (char *)
	     "7fffffff" "ffffffff" "e487ed51" "10b4611a" "62633145" "c06e0e68"
	     "94812704" "4533e63a" "0105df53" "1d89cd91" "28a5043c" "c71a026e"
	     "f7ca8cd9" "e69d218d" "98158536" "f92f8a1b" "a7f09ab6" "b6a8e122"
	     "f242dabb" "312f3f63" "7a262174" "d31bf6b5" "85ffae5b" "7a035bf6"
	     "f71c35fd" "ad44cfd2" "d74f9208" "be258ff3" "24943328" "f6722d9e"
	     "e1003e5c" "50b1df82" "cc6d241b" "0e2ae9cd" "348b1fd4" "7e9267af"
	     "c1b2ae91" "ee51d6cb" "0e3179ab" "1042a95d" "cf6a9483" "b84b4b36"
	     "b3861aa7" "255e4c02" "78ba3604" "650c10be" "19482f23" "171b671d"
	     "f1cf3b96" "0c074301" "cd93c1d1" "7603d147" "dae2aef8" "37a62964"
	     "ef15e5fb" "4aac0b8c" "1ccaa4be" "754ab572" "8ae9130c" "4c7d0288"
	     "0ab9472d" "45556216" "d6998b86" "82283d19" "d42a90d5" "ef8e5d32"
	     "767dc282" "2c6df785" "457538ab" "ae83063e" "d9cb87c2" "d370f263"
	     "d5fad746" "6d8499eb" "8f464a70" "2512b0ce" "e771e913" "0d697735"
	     "f897fd03" "6cc50432" "6c3b0139" "9f643532" "290f958c" "0bbd9006"
	     "5df08bab" "bd30aeb6" "3b84c460" "5d6ca371" "047127d0" "3a72d598"
	     "a1edadfe" "707e8847" "25c16890" "549d6965" "7fffffff" "ffffffff",
	.g = (char *) "2",
	.p_b = { NULL, 0, },
	.q_b = { NULL, 0, },
	.g_b = { NULL, 0, },
}, {
	/* rfc3526 group 16 "4096-bit MODP Group" */
	.p = (char *)
	     "FFFFFFFF" "FFFFFFFF" "C90FDAA2" "2168C234" "C4C6628B" "80DC1CD1"
	     "29024E08" "8A67CC74" "020BBEA6" "3B139B22" "514A0879" "8E3404DD"
	     "EF9519B3" "CD3A431B" "302B0A6D" "F25F1437" "4FE1356D" "6D51C245"
	     "E485B576" "625E7EC6" "F44C42E9" "A637ED6B" "0BFF5CB6" "F406B7ED"
	     "EE386BFB" "5A899FA5" "AE9F2411" "7C4B1FE6" "49286651" "ECE45B3D"
	     "C2007CB8" "A163BF05" "98DA4836" "1C55D39A" "69163FA8" "FD24CF5F"
	     "83655D23" "DCA3AD96" "1C62F356" "208552BB" "9ED52907" "7096966D"
	     "670C354E" "4ABC9804" "F1746C08" "CA18217C" "32905E46" "2E36CE3B"
	     "E39E772C" "180E8603" "9B2783A2" "EC07A28F" "B5C55DF0" "6F4C52C9"
	     "DE2BCBF6" "95581718" "3995497C" "EA956AE5" "15D22618" "98FA0510"
	     "15728E5A" "8AAAC42D" "AD33170D" "04507A33" "A85521AB" "DF1CBA64"
	     "ECFB8504" "58DBEF0A" "8AEA7157" "5D060C7D" "B3970F85" "A6E1E4C7"
	     "ABF5AE8C" "DB0933D7" "1E8C94E0" "4A25619D" "CEE3D226" "1AD2EE6B"
	     "F12FFA06" "D98A0864" "D8760273" "3EC86A64" "521F2B18" "177B200C"
	     "BBE11757" "7A615D6C" "770988C0" "BAD946E2" "08E24FA0" "74E5AB31"
	     "43DB5BFC" "E0FD108E" "4B82D120" "A9210801" "1A723C12" "A787E6D7"
	     "88719A10" "BDBA5B26" "99C32718" "6AF4E23C" "1A946834" "B6150BDA"
	     "2583E9CA" "2AD44CE8" "DBBBC2DB" "04DE8EF9" "2E8EFC14" "1FBECAA6"
	     "287C5947" "4E6BC05D" "99B2964F" "A090C3A2" "233BA186" "515BE7ED"
	     "1F612970" "CEE2D7AF" "B81BDD76" "2170481C" "D0069127" "D5B05AA9"
	     "93B4EA98" "8D8FDDC1" "86FFB7DC" "90A6C08F" "4DF435C9" "34063199"
	     "FFFFFFFF" "FFFFFFFF",
	.q = (char *)
	     "7fffffff" "ffffffff" "e487ed51" "10b4611a" "62633145" "c06e0e68"
	     "94812704" "4533e63a" "0105df53" "1d89cd91" "28a5043c" "c71a026e"
	     "f7ca8cd9" "e69d218d" "98158536" "f92f8a1b" "a7f09ab6" "b6a8e122"
	     "f242dabb" "312f3f63" "7a262174" "d31bf6b5" "85ffae5b" "7a035bf6"
	     "f71c35fd" "ad44cfd2" "d74f9208" "be258ff3" "24943328" "f6722d9e"
	     "e1003e5c" "50b1df82" "cc6d241b" "0e2ae9cd" "348b1fd4" "7e9267af"
	     "c1b2ae91" "ee51d6cb" "0e3179ab" "1042a95d" "cf6a9483" "b84b4b36"
	     "b3861aa7" "255e4c02" "78ba3604" "650c10be" "19482f23" "171b671d"
	     "f1cf3b96" "0c074301" "cd93c1d1" "7603d147" "dae2aef8" "37a62964"
	     "ef15e5fb" "4aac0b8c" "1ccaa4be" "754ab572" "8ae9130c" "4c7d0288"
	     "0ab9472d" "45556216" "d6998b86" "82283d19" "d42a90d5" "ef8e5d32"
	     "767dc282" "2c6df785" "457538ab" "ae83063e" "d9cb87c2" "d370f263"
	     "d5fad746" "6d8499eb" "8f464a70" "2512b0ce" "e771e913" "0d697735"
	     "f897fd03" "6cc50432" "6c3b0139" "9f643532" "290f958c" "0bbd9006"
	     "5df08bab" "bd30aeb6" "3b84c460" "5d6ca371" "047127d0" "3a72d598"
	     "a1edadfe" "707e8847" "25c16890" "54908400" "8d391e09" "53c3f36b"
	     "c438cd08" "5edd2d93" "4ce1938c" "357a711e" "0d4a341a" "5b0a85ed"
	     "12c1f4e5" "156a2674" "6ddde16d" "826f477c" "97477e0a" "0fdf6553"
	     "143e2ca3" "a735e02e" "ccd94b27" "d04861d1" "119dd0c3" "28adf3f6"
	     "8fb094b8" "67716bd7" "dc0deebb" "10b8240e" "68034893" "ead82d54"
	     "c9da754c" "46c7eee0" "c37fdbee" "48536047" "a6fa1ae4" "9a0318cc"
	     "ffffffff" "ffffffff",
	.g = (char *) "2",
	.p_b = { NULL, 0, },
	.q_b = { NULL, 0, },
	.g_b = { NULL, 0, },
}, {
	/* rfc3526 group 17 "6144-bit MODP Group" */
	.p = (char *)
	     "FFFFFFFF" "FFFFFFFF" "C90FDAA2" "2168C234" "C4C6628B" "80DC1CD1"
	     "29024E08" "8A67CC74" "020BBEA6" "3B139B22" "514A0879" "8E3404DD"
	     "EF9519B3" "CD3A431B" "302B0A6D" "F25F1437" "4FE1356D" "6D51C245"
	     "E485B576" "625E7EC6" "F44C42E9" "A637ED6B" "0BFF5CB6" "F406B7ED"
	     "EE386BFB" "5A899FA5" "AE9F2411" "7C4B1FE6" "49286651" "ECE45B3D"
	     "C2007CB8" "A163BF05" "98DA4836" "1C55D39A" "69163FA8" "FD24CF5F"
	     "83655D23" "DCA3AD96" "1C62F356" "208552BB" "9ED52907" "7096966D"
	     "670C354E" "4ABC9804" "F1746C08" "CA18217C" "32905E46" "2E36CE3B"
	     "E39E772C" "180E8603" "9B2783A2" "EC07A28F" "B5C55DF0" "6F4C52C9"
	     "DE2BCBF6" "95581718" "3995497C" "EA956AE5" "15D22618" "98FA0510"
	     "15728E5A" "8AAAC42D" "AD33170D" "04507A33" "A85521AB" "DF1CBA64"
	     "ECFB8504" "58DBEF0A" "8AEA7157" "5D060C7D" "B3970F85" "A6E1E4C7"
	     "ABF5AE8C" "DB0933D7" "1E8C94E0" "4A25619D" "CEE3D226" "1AD2EE6B"
	     "F12FFA06" "D98A0864" "D8760273" "3EC86A64" "521F2B18" "177B200C"
	     "BBE11757" "7A615D6C" "770988C0" "BAD946E2" "08E24FA0" "74E5AB31"
	     "43DB5BFC" "E0FD108E" "4B82D120" "A9210801" "1A723C12" "A787E6D7"
	     "88719A10" "BDBA5B26" "99C32718" "6AF4E23C" "1A946834" "B6150BDA"
	     "2583E9CA" "2AD44CE8" "DBBBC2DB" "04DE8EF9" "2E8EFC14" "1FBECAA6"
	     "287C5947" "4E6BC05D" "99B2964F" "A090C3A2" "233BA186" "515BE7ED"
	     "1F612970" "CEE2D7AF" "B81BDD76" "2170481C" "D0069127" "D5B05AA9"
	     "93B4EA98" "8D8FDDC1" "86FFB7DC" "90A6C08F" "4DF435C9" "34028492"
	     "36C3FAB4" "D27C7026" "C1D4DCB2" "602646DE" "C9751E76" "3DBA37BD"
	     "F8FF9406" "AD9E530E" "E5DB382F" "413001AE" "B06A53ED" "9027D831"
	     "179727B0" "865A8918" "DA3EDBEB" "CF9B14ED" "44CE6CBA" "CED4BB1B"
	     "DB7F1447" "E6CC254B" "33205151" "2BD7AF42" "6FB8F401" "378CD2BF"
	     "5983CA01" "C64B92EC" "F032EA15" "D1721D03" "F482D7CE" "6E74FEF6"
	     "D55E702F" "46980C82" "B5A84031" "900B1C9E" "59E7C97F" "BEC7E8F3"
	     "23A97A7E" "36CC88BE" "0F1D45B7" "FF585AC5" "4BD407B2" "2B4154AA"
	     "CC8F6D7E" "BF48E1D8" "14CC5ED2" "0F8037E0" "A79715EE" "F29BE328"
	     "06A1D58B" "B7C5DA76" "F550AA3D" "8A1FBFF0" "EB19CCB1" "A313D55C"
	     "DA56C9EC" "2EF29632" "387FE8D7" "6E3C0468" "043E8F66" "3F4860EE"
	     "12BF2D5B" "0B7474D6" "E694F91E" "6DCC4024" "FFFFFFFF" "FFFFFFFF",
	.q = (char *)
	     "7fffffff" "ffffffff" "e487ed51" "10b4611a" "62633145" "c06e0e68"
	     "94812704" "4533e63a" "0105df53" "1d89cd91" "28a5043c" "c71a026e"
	     "f7ca8cd9" "e69d218d" "98158536" "f92f8a1b" "a7f09ab6" "b6a8e122"
	     "f242dabb" "312f3f63" "7a262174" "d31bf6b5" "85ffae5b" "7a035bf6"
	     "f71c35fd" "ad44cfd2" "d74f9208" "be258ff3" "24943328" "f6722d9e"
	     "e1003e5c" "50b1df82" "cc6d241b" "0e2ae9cd" "348b1fd4" "7e9267af"
	     "c1b2ae91" "ee51d6cb" "0e3179ab" "1042a95d" "cf6a9483" "b84b4b36"
	     "b3861aa7" "255e4c02" "78ba3604" "650c10be" "19482f23" "171b671d"
	     "f1cf3b96" "0c074301" "cd93c1d1" "7603d147" "dae2aef8" "37a62964"
	     "ef15e5fb" "4aac0b8c" "1ccaa4be" "754ab572" "8ae9130c" "4c7d0288"
	     "0ab9472d" "45556216" "d6998b86" "82283d19" "d42a90d5" "ef8e5d32"
	     "767dc282" "2c6df785" "457538ab" "ae83063e" "d9cb87c2" "d370f263"
	     "d5fad746" "6d8499eb" "8f464a70" "2512b0ce" "e771e913" "0d697735"
	     "f897fd03" "6cc50432" "6c3b0139" "9f643532" "290f958c" "0bbd9006"
	     "5df08bab" "bd30aeb6" "3b84c460" "5d6ca371" "047127d0" "3a72d598"
	     "a1edadfe" "707e8847" "25c16890" "54908400" "8d391e09" "53c3f36b"
	     "c438cd08" "5edd2d93" "4ce1938c" "357a711e" "0d4a341a" "5b0a85ed"
	     "12c1f4e5" "156a2674" "6ddde16d" "826f477c" "97477e0a" "0fdf6553"
	     "143e2ca3" "a735e02e" "ccd94b27" "d04861d1" "119dd0c3" "28adf3f6"
	     "8fb094b8" "67716bd7" "dc0deebb" "10b8240e" "68034893" "ead82d54"
	     "c9da754c" "46c7eee0" "c37fdbee" "48536047" "a6fa1ae4" "9a014249"
	     "1b61fd5a" "693e3813" "60ea6e59" "3013236f" "64ba8f3b" "1edd1bde"
	     "fc7fca03" "56cf2987" "72ed9c17" "a09800d7" "583529f6" "c813ec18"
	     "8bcb93d8" "432d448c" "6d1f6df5" "e7cd8a76" "a267365d" "676a5d8d"
	     "edbf8a23" "f36612a5" "999028a8" "95ebd7a1" "37dc7a00" "9bc6695f"
	     "acc1e500" "e325c976" "7819750a" "e8b90e81" "fa416be7" "373a7f7b"
	     "6aaf3817" "a34c0641" "5ad42018" "c8058e4f" "2cf3e4bf" "df63f479"
	     "91d4bd3f" "1b66445f" "078ea2db" "ffac2d62" "a5ea03d9" "15a0aa55"
	     "6647b6bf" "5fa470ec" "0a662f69" "07c01bf0" "53cb8af7" "794df194"
	     "0350eac5" "dbe2ed3b" "7aa8551e" "c50fdff8" "758ce658" "d189eaae"
	     "6d2b64f6" "17794b19" "1c3ff46b" "b71e0234" "021f47b3" "1fa43077"
	     "095f96ad" "85ba3a6b" "734a7c8f" "36e62012" "7fffffff" "ffffffff",
	.g = (char *) "2",
	.p_b = { NULL, 0, },
	.q_b = { NULL, 0, },
	.g_b = { NULL, 0, },
}, {
	/* rfc3526 group 18 "8192-bit MODP Group" */
	.p = (char *)
	     "FFFFFFFF" "FFFFFFFF" "C90FDAA2" "2168C234" "C4C6628B" "80DC1CD1"
	     "29024E08" "8A67CC74" "020BBEA6" "3B139B22" "514A0879" "8E3404DD"
	     "EF9519B3" "CD3A431B" "302B0A6D" "F25F1437" "4FE1356D" "6D51C245"
	     "E485B576" "625E7EC6" "F44C42E9" "A637ED6B" "0BFF5CB6" "F406B7ED"
	     "EE386BFB" "5A899FA5" "AE9F2411" "7C4B1FE6" "49286651" "ECE45B3D"
	     "C2007CB8" "A163BF05" "98DA4836" "1C55D39A" "69163FA8" "FD24CF5F"
	     "83655D23" "DCA3AD96" "1C62F356" "208552BB" "9ED52907" "7096966D"
	     "670C354E" "4ABC9804" "F1746C08" "CA18217C" "32905E46" "2E36CE3B"
	     "E39E772C" "180E8603" "9B2783A2" "EC07A28F" "B5C55DF0" "6F4C52C9"
	     "DE2BCBF6" "95581718" "3995497C" "EA956AE5" "15D22618" "98FA0510"
	     "15728E5A" "8AAAC42D" "AD33170D" "04507A33" "A85521AB" "DF1CBA64"
	     "ECFB8504" "58DBEF0A" "8AEA7157" "5D060C7D" "B3970F85" "A6E1E4C7"
	     "ABF5AE8C" "DB0933D7" "1E8C94E0" "4A25619D" "CEE3D226" "1AD2EE6B"
	     "F12FFA06" "D98A0864" "D8760273" "3EC86A64" "521F2B18" "177B200C"
	     "BBE11757" "7A615D6C" "770988C0" "BAD946E2" "08E24FA0" "74E5AB31"
	     "43DB5BFC" "E0FD108E" "4B82D120" "A9210801" "1A723C12" "A787E6D7"
	     "88719A10" "BDBA5B26" "99C32718" "6AF4E23C" "1A946834" "B6150BDA"
	     "2583E9CA" "2AD44CE8" "DBBBC2DB" "04DE8EF9" "2E8EFC14" "1FBECAA6"
	     "287C5947" "4E6BC05D" "99B2964F" "A090C3A2" "233BA186" "515BE7ED"
	     "1F612970" "CEE2D7AF" "B81BDD76" "2170481C" "D0069127" "D5B05AA9"
	     "93B4EA98" "8D8FDDC1" "86FFB7DC" "90A6C08F" "4DF435C9" "34028492"
	     "36C3FAB4" "D27C7026" "C1D4DCB2" "602646DE" "C9751E76" "3DBA37BD"
	     "F8FF9406" "AD9E530E" "E5DB382F" "413001AE" "B06A53ED" "9027D831"
	     "179727B0" "865A8918" "DA3EDBEB" "CF9B14ED" "44CE6CBA" "CED4BB1B"
	     "DB7F1447" "E6CC254B" "33205151" "2BD7AF42" "6FB8F401" "378CD2BF"
	     "5983CA01" "C64B92EC" "F032EA15" "D1721D03" "F482D7CE" "6E74FEF6"
	     "D55E702F" "46980C82" "B5A84031" "900B1C9E" "59E7C97F" "BEC7E8F3"
	     "23A97A7E" "36CC88BE" "0F1D45B7" "FF585AC5" "4BD407B2" "2B4154AA"
	     "CC8F6D7E" "BF48E1D8" "14CC5ED2" "0F8037E0" "A79715EE" "F29BE328"
	     "06A1D58B" "B7C5DA76" "F550AA3D" "8A1FBFF0" "EB19CCB1" "A313D55C"
	     "DA56C9EC" "2EF29632" "387FE8D7" "6E3C0468" "043E8F66" "3F4860EE"
	     "12BF2D5B" "0B7474D6" "E694F91E" "6DBE1159" "74A3926F" "12FEE5E4"
	     "38777CB6" "A932DF8C" "D8BEC4D0" "73B931BA" "3BC832B6" "8D9DD300"
	     "741FA7BF" "8AFC47ED" "2576F693" "6BA42466" "3AAB639C" "5AE4F568"
	     "3423B474" "2BF1C978" "238F16CB" "E39D652D" "E3FDB8BE" "FC848AD9"
	     "22222E04" "A4037C07" "13EB57A8" "1A23F0C7" "3473FC64" "6CEA306B"
	     "4BCBC886" "2F8385DD" "FA9D4B7F" "A2C087E8" "79683303" "ED5BDD3A"
	     "062B3CF5" "B3A278A6" "6D2A13F8" "3F44F82D" "DF310EE0" "74AB6A36"
	     "4597E899" "A0255DC1" "64F31CC5" "0846851D" "F9AB4819" "5DED7EA1"
	     "B1D510BD" "7EE74D73" "FAF36BC3" "1ECFA268" "359046F4" "EB879F92"
	     "4009438B" "481C6CD7" "889A002E" "D5EE382B" "C9190DA6" "FC026E47"
	     "9558E447" "5677E9AA" "9E3050E2" "765694DF" "C81F56E8" "80B96E71"
	     "60C980DD" "98EDD3DF" "FFFFFFFF" "FFFFFFFF",
	.q = (char *)
	     "7fffffff" "ffffffff" "e487ed51" "10b4611a" "62633145" "c06e0e68"
	     "94812704" "4533e63a" "0105df53" "1d89cd91" "28a5043c" "c71a026e"
	     "f7ca8cd9" "e69d218d" "98158536" "f92f8a1b" "a7f09ab6" "b6a8e122"
	     "f242dabb" "312f3f63" "7a262174" "d31bf6b5" "85ffae5b" "7a035bf6"
	     "f71c35fd" "ad44cfd2" "d74f9208" "be258ff3" "24943328" "f6722d9e"
	     "e1003e5c" "50b1df82" "cc6d241b" "0e2ae9cd" "348b1fd4" "7e9267af"
	     "c1b2ae91" "ee51d6cb" "0e3179ab" "1042a95d" "cf6a9483" "b84b4b36"
	     "b3861aa7" "255e4c02" "78ba3604" "650c10be" "19482f23" "171b671d"
	     "f1cf3b96" "0c074301" "cd93c1d1" "7603d147" "dae2aef8" "37a62964"
	     "ef15e5fb" "4aac0b8c" "1ccaa4be" "754ab572" "8ae9130c" "4c7d0288"
	     "0ab9472d" "45556216" "d6998b86" "82283d19" "d42a90d5" "ef8e5d32"
	     "767dc282" "2c6df785" "457538ab" "ae83063e" "d9cb87c2" "d370f263"
	     "d5fad746" "6d8499eb" "8f464a70" "2512b0ce" "e771e913" "0d697735"
	     "f897fd03" "6cc50432" "6c3b0139" "9f643532" "290f958c" "0bbd9006"
	     "5df08bab" "bd30aeb6" "3b84c460" "5d6ca371" "047127d0" "3a72d598"
	     "a1edadfe" "707e8847" "25c16890" "54908400" "8d391e09" "53c3f36b"
	     "c438cd08" "5edd2d93" "4ce1938c" "357a711e" "0d4a341a" "5b0a85ed"
	     "12c1f4e5" "156a2674" "6ddde16d" "826f477c" "97477e0a" "0fdf6553"
	     "143e2ca3" "a735e02e" "ccd94b27" "d04861d1" "119dd0c3" "28adf3f6"
	     "8fb094b8" "67716bd7" "dc0deebb" "10b8240e" "68034893" "ead82d54"
	     "c9da754c" "46c7eee0" "c37fdbee" "48536047" "a6fa1ae4" "9a014249"
	     "1b61fd5a" "693e3813" "60ea6e59" "3013236f" "64ba8f3b" "1edd1bde"
	     "fc7fca03" "56cf2987" "72ed9c17" "a09800d7" "583529f6" "c813ec18"
	     "8bcb93d8" "432d448c" "6d1f6df5" "e7cd8a76" "a267365d" "676a5d8d"
	     "edbf8a23" "f36612a5" "999028a8" "95ebd7a1" "37dc7a00" "9bc6695f"
	     "acc1e500" "e325c976" "7819750a" "e8b90e81" "fa416be7" "373a7f7b"
	     "6aaf3817" "a34c0641" "5ad42018" "c8058e4f" "2cf3e4bf" "df63f479"
	     "91d4bd3f" "1b66445f" "078ea2db" "ffac2d62" "a5ea03d9" "15a0aa55"
	     "6647b6bf" "5fa470ec" "0a662f69" "07c01bf0" "53cb8af7" "794df194"
	     "0350eac5" "dbe2ed3b" "7aa8551e" "c50fdff8" "758ce658" "d189eaae"
	     "6d2b64f6" "17794b19" "1c3ff46b" "b71e0234" "021f47b3" "1fa43077"
	     "095f96ad" "85ba3a6b" "734a7c8f" "36df08ac" "ba51c937" "897f72f2"
	     "1c3bbe5b" "54996fc6" "6c5f6268" "39dc98dd" "1de4195b" "46cee980"
	     "3a0fd3df" "c57e23f6" "92bb7b49" "b5d21233" "1d55b1ce" "2d727ab4"
	     "1a11da3a" "15f8e4bc" "11c78b65" "f1ceb296" "f1fedc5f" "7e42456c"
	     "91111702" "5201be03" "89f5abd4" "0d11f863" "9a39fe32" "36751835"
	     "a5e5e443" "17c1c2ee" "fd4ea5bf" "d16043f4" "3cb41981" "f6adee9d"
	     "03159e7a" "d9d13c53" "369509fc" "1fa27c16" "ef988770" "3a55b51b"
	     "22cbf44c" "d012aee0" "b2798e62" "8423428e" "fcd5a40c" "aef6bf50"
	     "d8ea885e" "bf73a6b9" "fd79b5e1" "8f67d134" "1ac8237a" "75c3cfc9"
	     "2004a1c5" "a40e366b" "c44d0017" "6af71c15" "e48c86d3" "7e013723"
	     "caac7223" "ab3bf4d5" "4f182871" "3b2b4a6f" "e40fab74" "405cb738"
	     "b064c06e" "cc76e9ef" "ffffffff" "ffffffff",
	.g = (char *) "2",
	.p_b = { NULL, 0, },
	.q_b = { NULL, 0, },
	.g_b = { NULL, 0, },

}, {
	/* rfc71919 ffdhe2048 */
	.p = (char *)
	     "FFFFFFFF" "FFFFFFFF" "ADF85458" "A2BB4A9A" "AFDC5620" "273D3CF1"
	     "D8B9C583" "CE2D3695" "A9E13641" "146433FB" "CC939DCE" "249B3EF9"
	     "7D2FE363" "630C75D8" "F681B202" "AEC4617A" "D3DF1ED5" "D5FD6561"
	     "2433F51F" "5F066ED0" "85636555" "3DED1AF3" "B557135E" "7F57C935"
	     "984F0C70" "E0E68B77" "E2A689DA" "F3EFE872" "1DF158A1" "36ADE735"
	     "30ACCA4F" "483A797A" "BC0AB182" "B324FB61" "D108A94B" "B2C8E3FB"
	     "B96ADAB7" "60D7F468" "1D4F42A3" "DE394DF4" "AE56EDE7" "6372BB19"
	     "0B07A7C8" "EE0A6D70" "9E02FCE1" "CDF7E2EC" "C03404CD" "28342F61"
	     "9172FE9C" "E98583FF" "8E4F1232" "EEF28183" "C3FE3B1B" "4C6FAD73"
	     "3BB5FCBC" "2EC22005" "C58EF183" "7D1683B2" "C6F34A26" "C1B2EFFA"
	     "886B4238" "61285C97" "FFFFFFFF" "FFFFFFFF",
	.q = (char *)
	     "7FFFFFFF" "FFFFFFFF" "D6FC2A2C" "515DA54D" "57EE2B10" "139E9E78"
	     "EC5CE2C1" "E7169B4A" "D4F09B20" "8A3219FD" "E649CEE7" "124D9F7C"
	     "BE97F1B1" "B1863AEC" "7B40D901" "576230BD" "69EF8F6A" "EAFEB2B0"
	     "9219FA8F" "AF833768" "42B1B2AA" "9EF68D79" "DAAB89AF" "3FABE49A"
	     "CC278638" "707345BB" "F15344ED" "79F7F439" "0EF8AC50" "9B56F39A"
	     "98566527" "A41D3CBD" "5E0558C1" "59927DB0" "E88454A5" "D96471FD"
	     "DCB56D5B" "B06BFA34" "0EA7A151" "EF1CA6FA" "572B76F3" "B1B95D8C"
	     "8583D3E4" "770536B8" "4F017E70" "E6FBF176" "601A0266" "941A17B0"
	     "C8B97F4E" "74C2C1FF" "C7278919" "777940C1" "E1FF1D8D" "A637D6B9"
	     "9DDAFE5E" "17611002" "E2C778C1" "BE8B41D9" "6379A513" "60D977FD"
	     "4435A11C" "30942E4B" "FFFFFFFF" "FFFFFFFF",
	.g = (char *) "2",
	.p_b = { NULL, 0, },
	.q_b = { NULL, 0, },
	.g_b = { NULL, 0, },

}, {
	/* rfc71919 ffdhe3072 */
	.p = (char *)
	     "FFFFFFFF" "FFFFFFFF" "ADF85458" "A2BB4A9A" "AFDC5620" "273D3CF1"
	     "D8B9C583" "CE2D3695" "A9E13641" "146433FB" "CC939DCE" "249B3EF9"
	     "7D2FE363" "630C75D8" "F681B202" "AEC4617A" "D3DF1ED5" "D5FD6561"
	     "2433F51F" "5F066ED0" "85636555" "3DED1AF3" "B557135E" "7F57C935"
	     "984F0C70" "E0E68B77" "E2A689DA" "F3EFE872" "1DF158A1" "36ADE735"
	     "30ACCA4F" "483A797A" "BC0AB182" "B324FB61" "D108A94B" "B2C8E3FB"
	     "B96ADAB7" "60D7F468" "1D4F42A3" "DE394DF4" "AE56EDE7" "6372BB19"
	     "0B07A7C8" "EE0A6D70" "9E02FCE1" "CDF7E2EC" "C03404CD" "28342F61"
	     "9172FE9C" "E98583FF" "8E4F1232" "EEF28183" "C3FE3B1B" "4C6FAD73"
	     "3BB5FCBC" "2EC22005" "C58EF183" "7D1683B2" "C6F34A26" "C1B2EFFA"
	     "886B4238" "611FCFDC" "DE355B3B" "6519035B" "BC34F4DE" "F99C0238"
	     "61B46FC9" "D6E6C907" "7AD91D26" "91F7F7EE" "598CB0FA" "C186D91C"
	     "AEFE1309" "85139270" "B4130C93" "BC437944" "F4FD4452" "E2D74DD3"
	     "64F2E21E" "71F54BFF" "5CAE82AB" "9C9DF69E" "E86D2BC5" "22363A0D"
	     "ABC52197" "9B0DEADA" "1DBF9A42" "D5C4484E" "0ABCD06B" "FA53DDEF"
	     "3C1B20EE" "3FD59D7C" "25E41D2B" "66C62E37" "FFFFFFFF" "FFFFFFFF",
	.q = (char *)
	     "7FFFFFFF" "FFFFFFFF" "D6FC2A2C" "515DA54D" "57EE2B10" "139E9E78"
	     "EC5CE2C1" "E7169B4A" "D4F09B20" "8A3219FD" "E649CEE7" "124D9F7C"
	     "BE97F1B1" "B1863AEC" "7B40D901" "576230BD" "69EF8F6A" "EAFEB2B0"
	     "9219FA8F" "AF833768" "42B1B2AA" "9EF68D79" "DAAB89AF" "3FABE49A"
	     "CC278638" "707345BB" "F15344ED" "79F7F439" "0EF8AC50" "9B56F39A"
	     "98566527" "A41D3CBD" "5E0558C1" "59927DB0" "E88454A5" "D96471FD"
	     "DCB56D5B" "B06BFA34" "0EA7A151" "EF1CA6FA" "572B76F3" "B1B95D8C"
	     "8583D3E4" "770536B8" "4F017E70" "E6FBF176" "601A0266" "941A17B0"
	     "C8B97F4E" "74C2C1FF" "C7278919" "777940C1" "E1FF1D8D" "A637D6B9"
	     "9DDAFE5E" "17611002" "E2C778C1" "BE8B41D9" "6379A513" "60D977FD"
	     "4435A11C" "308FE7EE" "6F1AAD9D" "B28C81AD" "DE1A7A6F" "7CCE011C"
	     "30DA37E4" "EB736483" "BD6C8E93" "48FBFBF7" "2CC6587D" "60C36C8E"
	     "577F0984" "C289C938" "5A098649" "DE21BCA2" "7A7EA229" "716BA6E9"
	     "B279710F" "38FAA5FF" "AE574155" "CE4EFB4F" "743695E2" "911B1D06"
	     "D5E290CB" "CD86F56D" "0EDFCD21" "6AE22427" "055E6835" "FD29EEF7"
	     "9E0D9077" "1FEACEBE" "12F20E95" "B363171B" "FFFFFFFF" "FFFFFFFF",
	.g = (char *) "2",
	.p_b = { NULL, 0, },
	.q_b = { NULL, 0, },
	.g_b = { NULL, 0, },

}, {
	/* rfc71919 ffdhe4096 */
	.p = (char *)
	     "FFFFFFFF" "FFFFFFFF" "ADF85458" "A2BB4A9A" "AFDC5620" "273D3CF1"
	     "D8B9C583" "CE2D3695" "A9E13641" "146433FB" "CC939DCE" "249B3EF9"
	     "7D2FE363" "630C75D8" "F681B202" "AEC4617A" "D3DF1ED5" "D5FD6561"
	     "2433F51F" "5F066ED0" "85636555" "3DED1AF3" "B557135E" "7F57C935"
	     "984F0C70" "E0E68B77" "E2A689DA" "F3EFE872" "1DF158A1" "36ADE735"
	     "30ACCA4F" "483A797A" "BC0AB182" "B324FB61" "D108A94B" "B2C8E3FB"
	     "B96ADAB7" "60D7F468" "1D4F42A3" "DE394DF4" "AE56EDE7" "6372BB19"
	     "0B07A7C8" "EE0A6D70" "9E02FCE1" "CDF7E2EC" "C03404CD" "28342F61"
	     "9172FE9C" "E98583FF" "8E4F1232" "EEF28183" "C3FE3B1B" "4C6FAD73"
	     "3BB5FCBC" "2EC22005" "C58EF183" "7D1683B2" "C6F34A26" "C1B2EFFA"
	     "886B4238" "611FCFDC" "DE355B3B" "6519035B" "BC34F4DE" "F99C0238"
	     "61B46FC9" "D6E6C907" "7AD91D26" "91F7F7EE" "598CB0FA" "C186D91C"
	     "AEFE1309" "85139270" "B4130C93" "BC437944" "F4FD4452" "E2D74DD3"
	     "64F2E21E" "71F54BFF" "5CAE82AB" "9C9DF69E" "E86D2BC5" "22363A0D"
	     "ABC52197" "9B0DEADA" "1DBF9A42" "D5C4484E" "0ABCD06B" "FA53DDEF"
	     "3C1B20EE" "3FD59D7C" "25E41D2B" "669E1EF1" "6E6F52C3" "164DF4FB"
	     "7930E9E4" "E58857B6" "AC7D5F42" "D69F6D18" "7763CF1D" "55034004"
	     "87F55BA5" "7E31CC7A" "7135C886" "EFB4318A" "ED6A1E01" "2D9E6832"
	     "A907600A" "918130C4" "6DC778F9" "71AD0038" "092999A3" "33CB8B7A"
	     "1A1DB93D" "7140003C" "2A4ECEA9" "F98D0ACC" "0A8291CD" "CEC97DCF"
	     "8EC9B55A" "7F88A46B" "4DB5A851" "F44182E1" "C68A007E" "5E655F6A"
	     "FFFFFFFF" "FFFFFFFF",
	.q = (char *)
	     "7FFFFFFF" "FFFFFFFF" "D6FC2A2C" "515DA54D" "57EE2B10" "139E9E78"
	     "EC5CE2C1" "E7169B4A" "D4F09B20" "8A3219FD" "E649CEE7" "124D9F7C"
	     "BE97F1B1" "B1863AEC" "7B40D901" "576230BD" "69EF8F6A" "EAFEB2B0"
	     "9219FA8F" "AF833768" "42B1B2AA" "9EF68D79" "DAAB89AF" "3FABE49A"
	     "CC278638" "707345BB" "F15344ED" "79F7F439" "0EF8AC50" "9B56F39A"
	     "98566527" "A41D3CBD" "5E0558C1" "59927DB0" "E88454A5" "D96471FD"
	     "DCB56D5B" "B06BFA34" "0EA7A151" "EF1CA6FA" "572B76F3" "B1B95D8C"
	     "8583D3E4" "770536B8" "4F017E70" "E6FBF176" "601A0266" "941A17B0"
	     "C8B97F4E" "74C2C1FF" "C7278919" "777940C1" "E1FF1D8D" "A637D6B9"
	     "9DDAFE5E" "17611002" "E2C778C1" "BE8B41D9" "6379A513" "60D977FD"
	     "4435A11C" "308FE7EE" "6F1AAD9D" "B28C81AD" "DE1A7A6F" "7CCE011C"
	     "30DA37E4" "EB736483" "BD6C8E93" "48FBFBF7" "2CC6587D" "60C36C8E"
	     "577F0984" "C289C938" "5A098649" "DE21BCA2" "7A7EA229" "716BA6E9"
	     "B279710F" "38FAA5FF" "AE574155" "CE4EFB4F" "743695E2" "911B1D06"
	     "D5E290CB" "CD86F56D" "0EDFCD21" "6AE22427" "055E6835" "FD29EEF7"
	     "9E0D9077" "1FEACEBE" "12F20E95" "B34F0F78" "B737A961" "8B26FA7D"
	     "BC9874F2" "72C42BDB" "563EAFA1" "6B4FB68C" "3BB1E78E" "AA81A002"
	     "43FAADD2" "BF18E63D" "389AE443" "77DA18C5" "76B50F00" "96CF3419"
	     "5483B005" "48C09862" "36E3BC7C" "B8D6801C" "0494CCD1" "99E5C5BD"
	     "0D0EDC9E" "B8A0001E" "15276754" "FCC68566" "054148E6" "E764BEE7"
	     "C764DAAD" "3FC45235" "A6DAD428" "FA20C170" "E345003F" "2F32AFB5"
	     "7FFFFFFF" "FFFFFFFF",
	.g = (char *) "2",
	.p_b = { NULL, 0, },
	.q_b = { NULL, 0, },
	.g_b = { NULL, 0, },

}, {
	/* rfc71919 ffdhe6144 */
	.p = (char *)
	     "FFFFFFFF" "FFFFFFFF" "ADF85458" "A2BB4A9A" "AFDC5620" "273D3CF1"
	     "D8B9C583" "CE2D3695" "A9E13641" "146433FB" "CC939DCE" "249B3EF9"
	     "7D2FE363" "630C75D8" "F681B202" "AEC4617A" "D3DF1ED5" "D5FD6561"
	     "2433F51F" "5F066ED0" "85636555" "3DED1AF3" "B557135E" "7F57C935"
	     "984F0C70" "E0E68B77" "E2A689DA" "F3EFE872" "1DF158A1" "36ADE735"
	     "30ACCA4F" "483A797A" "BC0AB182" "B324FB61" "D108A94B" "B2C8E3FB"
	     "B96ADAB7" "60D7F468" "1D4F42A3" "DE394DF4" "AE56EDE7" "6372BB19"
	     "0B07A7C8" "EE0A6D70" "9E02FCE1" "CDF7E2EC" "C03404CD" "28342F61"
	     "9172FE9C" "E98583FF" "8E4F1232" "EEF28183" "C3FE3B1B" "4C6FAD73"
	     "3BB5FCBC" "2EC22005" "C58EF183" "7D1683B2" "C6F34A26" "C1B2EFFA"
	     "886B4238" "611FCFDC" "DE355B3B" "6519035B" "BC34F4DE" "F99C0238"
	     "61B46FC9" "D6E6C907" "7AD91D26" "91F7F7EE" "598CB0FA" "C186D91C"
	     "AEFE1309" "85139270" "B4130C93" "BC437944" "F4FD4452" "E2D74DD3"
	     "64F2E21E" "71F54BFF" "5CAE82AB" "9C9DF69E" "E86D2BC5" "22363A0D"
	     "ABC52197" "9B0DEADA" "1DBF9A42" "D5C4484E" "0ABCD06B" "FA53DDEF"
	     "3C1B20EE" "3FD59D7C" "25E41D2B" "669E1EF1" "6E6F52C3" "164DF4FB"
	     "7930E9E4" "E58857B6" "AC7D5F42" "D69F6D18" "7763CF1D" "55034004"
	     "87F55BA5" "7E31CC7A" "7135C886" "EFB4318A" "ED6A1E01" "2D9E6832"
	     "A907600A" "918130C4" "6DC778F9" "71AD0038" "092999A3" "33CB8B7A"
	     "1A1DB93D" "7140003C" "2A4ECEA9" "F98D0ACC" "0A8291CD" "CEC97DCF"
	     "8EC9B55A" "7F88A46B" "4DB5A851" "F44182E1" "C68A007E" "5E0DD902"
	     "0BFD64B6" "45036C7A" "4E677D2C" "38532A3A" "23BA4442" "CAF53EA6"
	     "3BB45432" "9B7624C8" "917BDD64" "B1C0FD4C" "B38E8C33" "4C701C3A"
	     "CDAD0657" "FCCFEC71" "9B1F5C3E" "4E46041F" "388147FB" "4CFDB477"
	     "A52471F7" "A9A96910" "B855322E" "DB6340D8" "A00EF092" "350511E3"
	     "0ABEC1FF" "F9E3A26E" "7FB29F8C" "183023C3" "587E38DA" "0077D9B4"
	     "763E4E4B" "94B2BBC1" "94C6651E" "77CAF992" "EEAAC023" "2A281BF6"
	     "B3A739C1" "22611682" "0AE8DB58" "47A67CBE" "F9C9091B" "462D538C"
	     "D72B0374" "6AE77F5E" "62292C31" "1562A846" "505DC82D" "B854338A"
	     "E49F5235" "C95B9117" "8CCF2DD5" "CACEF403" "EC9D1810" "C6272B04"
	     "5B3B71F9" "DC6B80D6" "3FDD4A8E" "9ADB1E69" "62A69526" "D43161C1"
	     "A41D570D" "7938DAD4" "A40E329C" "D0E40E65" "FFFFFFFF" "FFFFFFFF",
	.q = (char *)
	     "7FFFFFFF" "FFFFFFFF" "D6FC2A2C" "515DA54D" "57EE2B10" "139E9E78"
	     "EC5CE2C1" "E7169B4A" "D4F09B20" "8A3219FD" "E649CEE7" "124D9F7C"
	     "BE97F1B1" "B1863AEC" "7B40D901" "576230BD" "69EF8F6A" "EAFEB2B0"
	     "9219FA8F" "AF833768" "42B1B2AA" "9EF68D79" "DAAB89AF" "3FABE49A"
	     "CC278638" "707345BB" "F15344ED" "79F7F439" "0EF8AC50" "9B56F39A"
	     "98566527" "A41D3CBD" "5E0558C1" "59927DB0" "E88454A5" "D96471FD"
	     "DCB56D5B" "B06BFA34" "0EA7A151" "EF1CA6FA" "572B76F3" "B1B95D8C"
	     "8583D3E4" "770536B8" "4F017E70" "E6FBF176" "601A0266" "941A17B0"
	     "C8B97F4E" "74C2C1FF" "C7278919" "777940C1" "E1FF1D8D" "A637D6B9"
	     "9DDAFE5E" "17611002" "E2C778C1" "BE8B41D9" "6379A513" "60D977FD"
	     "4435A11C" "308FE7EE" "6F1AAD9D" "B28C81AD" "DE1A7A6F" "7CCE011C"
	     "30DA37E4" "EB736483" "BD6C8E93" "48FBFBF7" "2CC6587D" "60C36C8E"
	     "577F0984" "C289C938" "5A098649" "DE21BCA2" "7A7EA229" "716BA6E9"
	     "B279710F" "38FAA5FF" "AE574155" "CE4EFB4F" "743695E2" "911B1D06"
	     "D5E290CB" "CD86F56D" "0EDFCD21" "6AE22427" "055E6835" "FD29EEF7"
	     "9E0D9077" "1FEACEBE" "12F20E95" "B34F0F78" "B737A961" "8B26FA7D"
	     "BC9874F2" "72C42BDB" "563EAFA1" "6B4FB68C" "3BB1E78E" "AA81A002"
	     "43FAADD2" "BF18E63D" "389AE443" "77DA18C5" "76B50F00" "96CF3419"
	     "5483B005" "48C09862" "36E3BC7C" "B8D6801C" "0494CCD1" "99E5C5BD"
	     "0D0EDC9E" "B8A0001E" "15276754" "FCC68566" "054148E6" "E764BEE7"
	     "C764DAAD" "3FC45235" "A6DAD428" "FA20C170" "E345003F" "2F06EC81"
	     "05FEB25B" "2281B63D" "2733BE96" "1C29951D" "11DD2221" "657A9F53"
	     "1DDA2A19" "4DBB1264" "48BDEEB2" "58E07EA6" "59C74619" "A6380E1D"
	     "66D6832B" "FE67F638" "CD8FAE1F" "2723020F" "9C40A3FD" "A67EDA3B"
	     "D29238FB" "D4D4B488" "5C2A9917" "6DB1A06C" "50077849" "1A8288F1"
	     "855F60FF" "FCF1D137" "3FD94FC6" "0C1811E1" "AC3F1C6D" "003BECDA"
	     "3B1F2725" "CA595DE0" "CA63328F" "3BE57CC9" "77556011" "95140DFB"
	     "59D39CE0" "91308B41" "05746DAC" "23D33E5F" "7CE4848D" "A316A9C6"
	     "6B9581BA" "3573BFAF" "31149618" "8AB15423" "282EE416" "DC2A19C5"
	     "724FA91A" "E4ADC88B" "C66796EA" "E5677A01" "F64E8C08" "63139582"
	     "2D9DB8FC" "EE35C06B" "1FEEA547" "4D6D8F34" "B1534A93" "6A18B0E0"
	     "D20EAB86" "BC9C6D6A" "5207194E" "68720732" "FFFFFFFF" "FFFFFFFF",
	.g = (char *) "2",
	.p_b = { NULL, 0, },
	.q_b = { NULL, 0, },
	.g_b = { NULL, 0, },

}, {
	/* rfc71919 ffdhe8192 */
	.p = (char *)
	     "FFFFFFFF" "FFFFFFFF" "ADF85458" "A2BB4A9A" "AFDC5620" "273D3CF1"
	     "D8B9C583" "CE2D3695" "A9E13641" "146433FB" "CC939DCE" "249B3EF9"
	     "7D2FE363" "630C75D8" "F681B202" "AEC4617A" "D3DF1ED5" "D5FD6561"
	     "2433F51F" "5F066ED0" "85636555" "3DED1AF3" "B557135E" "7F57C935"
	     "984F0C70" "E0E68B77" "E2A689DA" "F3EFE872" "1DF158A1" "36ADE735"
	     "30ACCA4F" "483A797A" "BC0AB182" "B324FB61" "D108A94B" "B2C8E3FB"
	     "B96ADAB7" "60D7F468" "1D4F42A3" "DE394DF4" "AE56EDE7" "6372BB19"
	     "0B07A7C8" "EE0A6D70" "9E02FCE1" "CDF7E2EC" "C03404CD" "28342F61"
	     "9172FE9C" "E98583FF" "8E4F1232" "EEF28183" "C3FE3B1B" "4C6FAD73"
	     "3BB5FCBC" "2EC22005" "C58EF183" "7D1683B2" "C6F34A26" "C1B2EFFA"
	     "886B4238" "611FCFDC" "DE355B3B" "6519035B" "BC34F4DE" "F99C0238"
	     "61B46FC9" "D6E6C907" "7AD91D26" "91F7F7EE" "598CB0FA" "C186D91C"
	     "AEFE1309" "85139270" "B4130C93" "BC437944" "F4FD4452" "E2D74DD3"
	     "64F2E21E" "71F54BFF" "5CAE82AB" "9C9DF69E" "E86D2BC5" "22363A0D"
	     "ABC52197" "9B0DEADA" "1DBF9A42" "D5C4484E" "0ABCD06B" "FA53DDEF"
	     "3C1B20EE" "3FD59D7C" "25E41D2B" "669E1EF1" "6E6F52C3" "164DF4FB"
	     "7930E9E4" "E58857B6" "AC7D5F42" "D69F6D18" "7763CF1D" "55034004"
	     "87F55BA5" "7E31CC7A" "7135C886" "EFB4318A" "ED6A1E01" "2D9E6832"
	     "A907600A" "918130C4" "6DC778F9" "71AD0038" "092999A3" "33CB8B7A"
	     "1A1DB93D" "7140003C" "2A4ECEA9" "F98D0ACC" "0A8291CD" "CEC97DCF"
	     "8EC9B55A" "7F88A46B" "4DB5A851" "F44182E1" "C68A007E" "5E0DD902"
	     "0BFD64B6" "45036C7A" "4E677D2C" "38532A3A" "23BA4442" "CAF53EA6"
	     "3BB45432" "9B7624C8" "917BDD64" "B1C0FD4C" "B38E8C33" "4C701C3A"
	     "CDAD0657" "FCCFEC71" "9B1F5C3E" "4E46041F" "388147FB" "4CFDB477"
	     "A52471F7" "A9A96910" "B855322E" "DB6340D8" "A00EF092" "350511E3"
	     "0ABEC1FF" "F9E3A26E" "7FB29F8C" "183023C3" "587E38DA" "0077D9B4"
	     "763E4E4B" "94B2BBC1" "94C6651E" "77CAF992" "EEAAC023" "2A281BF6"
	     "B3A739C1" "22611682" "0AE8DB58" "47A67CBE" "F9C9091B" "462D538C"
	     "D72B0374" "6AE77F5E" "62292C31" "1562A846" "505DC82D" "B854338A"
	     "E49F5235" "C95B9117" "8CCF2DD5" "CACEF403" "EC9D1810" "C6272B04"
	     "5B3B71F9" "DC6B80D6" "3FDD4A8E" "9ADB1E69" "62A69526" "D43161C1"
	     "A41D570D" "7938DAD4" "A40E329C" "CFF46AAA" "36AD004C" "F600C838"
	     "1E425A31" "D951AE64" "FDB23FCE" "C9509D43" "687FEB69" "EDD1CC5E"
	     "0B8CC3BD" "F64B10EF" "86B63142" "A3AB8829" "555B2F74" "7C932665"
	     "CB2C0F1C" "C01BD702" "29388839" "D2AF05E4" "54504AC7" "8B758282"
	     "2846C0BA" "35C35F5C" "59160CC0" "46FD8251" "541FC68C" "9C86B022"
	     "BB709987" "6A460E74" "51A8A931" "09703FEE" "1C217E6C" "3826E52C"
	     "51AA691E" "0E423CFC" "99E9E316" "50C1217B" "624816CD" "AD9A95F9"
	     "D5B80194" "88D9C0A0" "A1FE3075" "A577E231" "83F81D4A" "3F2FA457"
	     "1EFC8CE0" "BA8A4FE8" "B6855DFE" "72B0A66E" "DED2FBAB" "FBE58A30"
	     "FAFABE1C" "5D71A87E" "2F741EF8" "C1FE86FE" "A6BBFDE5" "30677F0D"
	     "97D11D49" "F7A8443D" "0822E506" "A9F4614E" "011E2A94" "838FF88C"
	     "D68C8BB7" "C5C6424C" "FFFFFFFF" "FFFFFFFF",
	.q = (char *)
	     "7FFFFFFF" "FFFFFFFF" "D6FC2A2C" "515DA54D" "57EE2B10" "139E9E78"
	     "EC5CE2C1" "E7169B4A" "D4F09B20" "8A3219FD" "E649CEE7" "124D9F7C"
	     "BE97F1B1" "B1863AEC" "7B40D901" "576230BD" "69EF8F6A" "EAFEB2B0"
	     "9219FA8F" "AF833768" "42B1B2AA" "9EF68D79" "DAAB89AF" "3FABE49A"
	     "CC278638" "707345BB" "F15344ED" "79F7F439" "0EF8AC50" "9B56F39A"
	     "98566527" "A41D3CBD" "5E0558C1" "59927DB0" "E88454A5" "D96471FD"
	     "DCB56D5B" "B06BFA34" "0EA7A151" "EF1CA6FA" "572B76F3" "B1B95D8C"
	     "8583D3E4" "770536B8" "4F017E70" "E6FBF176" "601A0266" "941A17B0"
	     "C8B97F4E" "74C2C1FF" "C7278919" "777940C1" "E1FF1D8D" "A637D6B9"
	     "9DDAFE5E" "17611002" "E2C778C1" "BE8B41D9" "6379A513" "60D977FD"
	     "4435A11C" "308FE7EE" "6F1AAD9D" "B28C81AD" "DE1A7A6F" "7CCE011C"
	     "30DA37E4" "EB736483" "BD6C8E93" "48FBFBF7" "2CC6587D" "60C36C8E"
	     "577F0984" "C289C938" "5A098649" "DE21BCA2" "7A7EA229" "716BA6E9"
	     "B279710F" "38FAA5FF" "AE574155" "CE4EFB4F" "743695E2" "911B1D06"
	     "D5E290CB" "CD86F56D" "0EDFCD21" "6AE22427" "055E6835" "FD29EEF7"
	     "9E0D9077" "1FEACEBE" "12F20E95" "B34F0F78" "B737A961" "8B26FA7D"
	     "BC9874F2" "72C42BDB" "563EAFA1" "6B4FB68C" "3BB1E78E" "AA81A002"
	     "43FAADD2" "BF18E63D" "389AE443" "77DA18C5" "76B50F00" "96CF3419"
	     "5483B005" "48C09862" "36E3BC7C" "B8D6801C" "0494CCD1" "99E5C5BD"
	     "0D0EDC9E" "B8A0001E" "15276754" "FCC68566" "054148E6" "E764BEE7"
	     "C764DAAD" "3FC45235" "A6DAD428" "FA20C170" "E345003F" "2F06EC81"
	     "05FEB25B" "2281B63D" "2733BE96" "1C29951D" "11DD2221" "657A9F53"
	     "1DDA2A19" "4DBB1264" "48BDEEB2" "58E07EA6" "59C74619" "A6380E1D"
	     "66D6832B" "FE67F638" "CD8FAE1F" "2723020F" "9C40A3FD" "A67EDA3B"
	     "D29238FB" "D4D4B488" "5C2A9917" "6DB1A06C" "50077849" "1A8288F1"
	     "855F60FF" "FCF1D137" "3FD94FC6" "0C1811E1" "AC3F1C6D" "003BECDA"
	     "3B1F2725" "CA595DE0" "CA63328F" "3BE57CC9" "77556011" "95140DFB"
	     "59D39CE0" "91308B41" "05746DAC" "23D33E5F" "7CE4848D" "A316A9C6"
	     "6B9581BA" "3573BFAF" "31149618" "8AB15423" "282EE416" "DC2A19C5"
	     "724FA91A" "E4ADC88B" "C66796EA" "E5677A01" "F64E8C08" "63139582"
	     "2D9DB8FC" "EE35C06B" "1FEEA547" "4D6D8F34" "B1534A93" "6A18B0E0"
	     "D20EAB86" "BC9C6D6A" "5207194E" "67FA3555" "1B568026" "7B00641C"
	     "0F212D18" "ECA8D732" "7ED91FE7" "64A84EA1" "B43FF5B4" "F6E8E62F"
	     "05C661DE" "FB258877" "C35B18A1" "51D5C414" "AAAD97BA" "3E499332"
	     "E596078E" "600DEB81" "149C441C" "E95782F2" "2A282563" "C5BAC141"
	     "1423605D" "1AE1AFAE" "2C8B0660" "237EC128" "AA0FE346" "4E435811"
	     "5DB84CC3" "B523073A" "28D45498" "84B81FF7" "0E10BF36" "1C137296"
	     "28D5348F" "07211E7E" "4CF4F18B" "286090BD" "B1240B66" "D6CD4AFC"
	     "EADC00CA" "446CE050" "50FF183A" "D2BBF118" "C1FC0EA5" "1F97D22B"
	     "8F7E4670" "5D4527F4" "5B42AEFF" "39585337" "6F697DD5" "FDF2C518"
	     "7D7D5F0E" "2EB8D43F" "17BA0F7C" "60FF437F" "535DFEF2" "9833BF86"
	     "CBE88EA4" "FBD4221E" "84117283" "54FA30A7" "008F154A" "41C7FC46"
	     "6B4645DB" "E2E32126" "7FFFFFFF" "FFFFFFFF",
	.g = (char *) "2",
	.p_b = { NULL, 0, },
	.q_b = { NULL, 0, },
	.g_b = { NULL, 0, },
} };

static void acvp_safeprime_clear(void)
{
	unsigned int i;

	for (i = 0; i < ARRAY_SIZE(safeprimes); i++) {
		free_buf(&safeprimes[i].p_b);
		free_buf(&safeprimes[i].q_b);
		free_buf(&safeprimes[i].g_b);
	}
}

static inline int acvp_safeprime_get(uint64_t safeprime,
				     struct safeprimes **out)
{
	struct safeprimes *s;
	static int atexit_set = 0;
	int ret = 0;

	switch (safeprime) {
	case ACVP_DH_MODP_2048:
		s = &safeprimes[0];
		break;
	case ACVP_DH_MODP_3072:
		s = &safeprimes[1];
		break;
	case ACVP_DH_MODP_4096:
		s = &safeprimes[2];
		break;
	case ACVP_DH_MODP_6144:
		s = &safeprimes[3];
		break;
	case ACVP_DH_MODP_8192:
		s = &safeprimes[4];
		break;
	case ACVP_DH_FFDHE_2048:
		s = &safeprimes[5];
		break;
	case ACVP_DH_FFDHE_3072:
		s = &safeprimes[6];
		break;
	case ACVP_DH_FFDHE_4096:
		s = &safeprimes[7];
		break;
	case ACVP_DH_FFDHE_6144:
		s = &safeprimes[8];
		break;
	case ACVP_DH_FFDHE_8192:
		s = &safeprimes[9];
		break;
	default:
		logger(LOGGER_ERR, "Unknown safeprime reference\n");
		return -EINVAL;
	}

	if (!s->p_b.buf) {
		CKINT(hex2bin_alloc(s->p, strlen(s->p), &s->p_b.buf,
				    &s->p_b.len));
	}
	if (!s->q_b.buf) {
		CKINT(hex2bin_alloc(s->q, strlen(s->q), &s->q_b.buf,
				    &s->q_b.len));
	}
	if (!s->g_b.buf) {
		CKINT(hex2bin_alloc(s->g, strlen(s->g), &s->g_b.buf,
				    &s->g_b.len));
	}

	if (!atexit_set) {
		atexit(acvp_safeprime_clear);
		atexit_set = 1;
	}

	*out = s;

out:
	return ret;
}

#ifdef __cplusplus
}
#endif

#endif /* SAFEPRIMES_H */
