#!/bin/bash
#
# CAVS test executor
# Written and Copyright (c) by: Stephan Müller <smueller@chronox.de>
#
# License: see LICENSE file
#
#                            NO WARRANTY
#
#    BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
#    FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN
#    OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
#    PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED
#    OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
#    MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS
#    TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE
#    PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,
#    REPAIR OR CORRECTION.
#
#    IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
#    WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
#    REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,
#    INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING
#    OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED
#    TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY
#    YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER
#    PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE
#    POSSIBILITY OF SUCH DAMAGES.

. $(dirname $0)/exec_lib.sh

TARGET="openssl"

MODULE_PREFIX="OpenSSL_"
MODULE_POSTFIX="_"

EXEC_TYPES_DRBG10X__64_bit___="CFLAGS=-DOPENSSL_DRBG_10X"
EXEC_TYPES_DRBG10X__32_bit___="CFLAGS=\"-m32 -DOPENSSL_DRBG_10X\" LDFLAGS=-m32"

EXEC="TDES_C FFC_DH KBKDF TLS_v1.3"

# Environment variable evaluated by ACVP Parser
CIPHER_CALL_FFC_DH="OPENSSL_ACVP_DH_KEYGEN=1"

if [ $(uname -m) = "s390x" ]; then
	EXEC="$EXEC
	      SHA_ASM SHA3_ASM SSH_ASM AESASM AESASM_ASM"
elif [ $(uname -m) = "aarch64" ]; then
	EXEC="$EXEC
	      SHA_ASM SHA3_ASM SSH_ASM
	      AES_C AES_C_GCM
	      NEON VPAES VPAES_GCM CE CE_GCM SHA_CE SHA3_CE"

	# Unlike for x86, OPENSSL_armcap_P is taken at face value
	# define ARMV7_NEON      (1<<0)
	# define ARMV7_TICK      (1<<1)
	# define ARMV8_AES       (1<<2)
	# define ARMV8_SHA1      (1<<3)
	# define ARMV8_SHA256    (1<<4)
	# define ARMV8_PMULL     (1<<5)
	# define ARMV8_SHA512    (1<<6)
	CIPHER_CALL_TDES_C="OPENSSL_armcap_P=0"
	CIPHER_CALL_AES_C_GCM="OPENSSL_armcap_P=0"
	CIPHER_CALL_AES_C="OPENSSL_armcap_P=0"
	CIPHER_CALL_SHA_ASM="OPENSSL_armcap_P=0"
	CIPHER_CALL_SHA3_ASM="OPENSSL_armcap_P=0"

	# Set the NEON bit
	CIPHER_CALL_VPAES="OPENSSL_armcap_P=1"
	CIPHER_CALL_SHA_NEON="OPENSSL_armcap_P=1"

	# Set the CE bits
	CIPHER_CALL_CE="OPENSSL_armcap_P=7D"
	CIPHER_CALL_CE_GCM="OPENSSL_armcap_P=7D"
	CIPHER_CALL_SHA_CE="OPENSSL_armcap_P=7D"
	CIPHER_CALL_SHA3_CE="OPENSSL_armcap_P=7D"
else
	EXEC="$EXEC
	      AESNI AESNI_AVX AESNI_CLMULNI AESNI_ASM
	      AESASM AESASM_AVX AESASM_CLMULNI AESASM_ASM
              BAES_CTASM BAES_CTASM_AVX BAES_CTASM_CLMULNI BAES_CTASM_ASM
              SHA_AVX2 SHA_AVX SHA_SSSE3 SHA_ASM SHA3_AVX2 SHA3_ASM SHA3_AVX512
              SSH_AVX2 SSH_AVX SSH_SSSE3 SSH_ASM"

	OPENSSL_REMOVE_AVX_SHA=":~0xe0200020"						# remove bits #64+21/29/30/31 (SHA512), #64+5 (AVX2)
	OPENSSL_REMOVE_AVX=":~0x20"							# remove bit #64+5 (AVX2)


	# Assembler cipher and C block chaining modes
	CIPHER_CALL_TDES_C=""

	CIPHER_CALL_AESNI=""								# used by default

	CIPHER_CALL_AESNI_AVX="OPENSSL_ia32cap=~0x0${OPENSSL_REMOVE_AVX_SHA}"		# remove bits #64+21/29/30/31 (SHA512) , #64+5 (AVX2)

	CIPHER_CALL_AESNI_CLMULNI="OPENSSL_ia32cap=~0x1000000000000000${OPENSSL_REMOVE_AVX}"		# remove bit #64+5 (AVX2), 60 (AVX) - AVX takes precedence over CLMUL for GHASH, so clear the AVX bit here.

	CIPHER_CALL_AESNI_ASM="OPENSSL_ia32cap=~0x01000000:0x0"				# remove bit 24 (XMM)

	CIPHER_CALL_AESASM="OPENSSL_ia32cap=~0x200020000000000:0x0"			# remove bits 57 (AESNI), 41 (SSSE3) - clear SSSE3 as BAES_CTASM would takes precedence otherwise

	CIPHER_CALL_AESASM_AVX="OPENSSL_ia32cap=~0x200020000000000${OPENSSL_REMOVE_AVX_SHA}"	# remove bits #64+21/29/30/31 (SHA512) , #64+5 (AVX2), 57 (AESNI), 41 (SSSE3) - clear SSSE3 as BAES_CTASM would takes precedence otherwise

	CIPHER_CALL_AESASM_CLMULNI="OPENSSL_ia32cap=~0x300020000000000${OPENSSL_REMOVE_AVX}"	# remove bits #64+5 (AVX2), 60 (AVX), 57 (AESNI), 41 (SSSE3) - AVX takes precedence over CLMUL for GHASH, so clear the AVX bit here.

	CIPHER_CALL_AESASM_ASM="OPENSSL_ia32cap=~0x300020200000000${OPENSSL_REMOVE_AVX}"	# remove bits #64+5 (AVX2), 60 (AVX), 57 (AESNI), 41 (SSSE3), 33 (CLMULNI) - AVX takes precedence over CLMUL for GHASH, so clear the AVX bit here.

	CIPHER_CALL_BAES_CTASM="OPENSSL_ia32cap=~0x200020200000000${OPENSSL_REMOVE_AVX_SHA}"	# remove bits #64+21/29/30/31 (SHA512) , #64+5 (AVX2), 57 (AESNI), 41 (SSSE3), 33 (CLMULNI)

	CIPHER_CALL_BAES_CTASM_AVX="OPENSSL_ia32cap=~0x200000000000000${OPENSSL_REMOVE_AVX}"	# remove bits #64+5 (AVX2), 57 (AESNI)

	CIPHER_CALL_BAES_CTASM_CLMULNI="OPENSSL_ia32cap=~0x200000000000000${OPENSSL_REMOVE_AVX}"	# remove bits #64+5 (AVX2), 57 (AESNI)

	CIPHER_CALL_BAES_CTASM_ASM="OPENSSL_ia32cap=~0x200000200000000${OPENSSL_REMOVE_AVX}"		# remove bits #64+5 (AVX2), 57 (AESNI), bit 33 (CLMULNI)

	CIPHER_CALL_SHA3_AVX512=""							# used by default if support found)
	CIPHER_CALL_SHA_AVX2="OPENSSL_ia32cap=${OPENSSL_REMOVE_AVX_SHA}"		# remove bits #64+21/29/30/31 (SHA512)
	CIPHER_CALL_SHA3_AVX2=$CIPHER_CALL_SHA_AVX2
	CIPHER_CALL_SHA_AVX="OPENSSL_ia32cap=~0x0${OPENSSL_REMOVE_AVX_SHA}"		# remove bits #64+21/29/30/31 (SHA512) , #64+5 (AVX2)
	CIPHER_CALL_SHA_SSSE3="OPENSSL_ia32cap=~0x1000000000000000${OPENSSL_REMOVE_AVX_SHA}"	# remove bits #64+21/29/30/31 (SHA512) , #64+5 (AVX2), 60 (AVX)
	CIPHER_CALL_SHA_ASM="OPENSSL_ia32cap=~0x1000020000000000${OPENSSL_REMOVE_AVX_SHA}"	# remove bits #64+21/29/30/31 (SHA512) , #64+5 (AVX2), 60 (AVX), 41 (SSSE3)
	CIPHER_CALL_SHA3_ASM=$CIPHER_CALL_SHA_ASM

fi

################### Heavy Lifting #######################

do_test() {
	PATH=.:$PATH

	for type in $EXEC_TYPES; do
		eval BUILD_FLAGS=\$EXEC_TYPES_$type

		eval "$BUILD_FLAGS build_tool ${TARGET}"

		for exec in $EXEC; do
			eval CIPHER_CALL=\${CIPHER_CALL_${exec}} 2> /dev/null

			local modulename="${MODULE_PREFIX}${type}${exec}${MODULE_POSTFIX}"

			eval "$CIPHER_CALL exec_module ${modulename}"
		done

		clean_tool
	done
}

do_test_drbg10x() {
	PATH=.:$PATH

	for type in $EXEC_TYPES; do
		eval BUILD_FLAGS=\$EXEC_TYPES_DRBG10X_$type

		eval "$BUILD_FLAGS build_tool ${TARGET}"

		local modulename="${MODULE_PREFIX}${type}DRBG_10X${MODULE_POSTFIX}"

		eval "exec_module ${modulename}"

		clean_tool
	done
}

do_test
do_test_drbg10x
exit $failures
