/*
 * Copyright (C) 2020, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef PARSER_IFC_H
#define PARSER_IFC_H

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @brief SP800-56B rev 2 KTS IFC data structure for the initiator algorithm
 *	  functional test (AFT) - a form of known-answer test.
 *
 * @var n [in] RSA N parameter of the key
 * @var e [in] RSA exponent of the key
 * @var iut_c [out] The ciphertext computed by the IUT (using the ACVP server's
 *	public key)
 * @var dkm [out] The derived keying material as a result of the KTS negotiation
 * @var tag [out] The key confirmation resulting MAC tag from the perspective
 *	of the IUT
 */
struct kts_ifc_init_data {
	struct buffer n;
	struct buffer e;

	struct buffer iut_c;
	struct buffer dkm;
	struct buffer tag;
};

/**
 * @brief SP800-56B rev 2 KTS IFC data structure for the initiator validation
 *	  testing (VAL). The IUT should verify the provided data and return
 *	  the status whether the KTS negotiation would have succeeded.
 *
 * @var n [in] RSA N parameter of the key
 * @var e [in] RSA exponent of the key
 * @var p [in] RSA P parameter of the key - may be NULL
 * @var q [in] RSA Q parameter of the key - may be NULL
 * @var dkm [in] The derived keying material to be validated
 * @val validation_success [out] The KTS negotiation has succeeded (1) or not
 *	succeeded (0)
 */
struct kts_ifc_init_validation_data {
	struct buffer n;
	struct buffer e;
	struct buffer p;
	struct buffer q;
	struct buffer dkm;

	uint32_t validation_success;
};

/**
 * @brief SP800-56B rev 2 KTS data structure for the responder algorithm
 *	  functional test (AFT) - a form of known-answer test.
 *
 * @var n [in] RSA N parameter of the key
 * @var e [in] RSA exponent of the key
 * @var p [in] RSA P parameter of the key
 * @var q [in] RSA Q parameter of the key
 * @var d [in] RSA D parameter of the key
 * @var c [in] The ciphertext generated by the ACVP server, encrypted with the
 *	IUT's public key
 * @var dkm [out] The derived keying material as a result of the KTS negotiation
 * @var tag [out] The key confirmation resulting MAC tag from the perspective
 *	of the IUT
 */
struct kts_ifc_resp_data {
	struct buffer n;
	struct buffer e;
	struct buffer p;
	struct buffer q;
	struct buffer d;
	struct buffer c;

	struct buffer dkm;
	struct buffer tag;
};

/**
 * @brief SP800-56B rev 2 KTS data structure for the responder validation
 *	  testing (VAL). The IUT should verify the provided data and return
 *	  the status whether the KTS negotiation would have succeeded.
 *
 * @var n [in] RSA N parameter of the key
 * @var e [in] RSA exponent of the key
 * @var p [in] RSA P parameter of the key
 * @var q [in] RSA Q parameter of the key
 * @var d [in] RSA D parameter of the key
 * @var c [in] The ciphertext generated by the ACVP server, encrypted with the
 *	IUT's public key
 * @var dkm [in] The derived keying material as a result of the KTS negotiation
 * @val validation_success [out] The KTS negotiation has succeeded (1) or not
 *	succeeded (0)
 */
struct kts_ifc_resp_validation_data {
	struct buffer n;
	struct buffer e;
	struct buffer p;
	struct buffer q;
	struct buffer d;
	struct buffer c;
	struct buffer dkm;

	uint32_t validation_success;
};

/**
 * @brief SP800-56B rev 2 KTS data structure holding the common test definition.
 *	  All options except for the union are applicable to the KTS IFC
 *	  testing.
 *
 * The union defined in this data structure fills the test-specific data. The
 * backend knows the type of testing and thus the data structure from the union
 * based on the following check:
 *
 *	if ((parsed_flags & FLAG_OP_KAS_ROLE_INITIATOR) &&
 *	    (parsed_flags & FLAG_OP_AFT)) {
 *		struct kts_ifc_init_data *init = &data->u.kts_ifc_init;
 *
 *		// KTS IFC initiator AFT testing
 *
 *	} else if ((parsed_flags & FLAG_OP_KAS_ROLE_RESPONDER) &&
 *		   (parsed_flags & FLAG_OP_AFT)) {
 *		struct kts_ifc_resp_data *resp = &data->u.kts_ifc_resp;
 *
 *		// KTS IFC responder AFT testing
 *
 *	} else if ((parsed_flags & FLAG_OP_KAS_ROLE_INITIATOR) &&
 *		   (parsed_flags & FLAG_OP_VAL)) {
 *		struct kts_ifc_init_validation_data *init_val =
 *					&data->u.kts_ifc_init_validation;
 *
 *		// KTS IFC initiator VAL testing
 *
 *	} else if ((parsed_flags & FLAG_OP_KAS_ROLE_RESPONDER) &&
 *		   (parsed_flags & FLAG_OP_VAL)) {
 *		struct kts_ifc_resp_validation_data *resp_val =
 *					&data->u.kts_ifc_resp_validation;
 *
 *		// KTS IFC responder VAL testing
 *
 *	} else {
 *		logger(LOGGER_ERR, "Unknown test\n");
 *		ret = -EINVAL;
 *	}
 *
 * @val schema [in] The scheme in use for the testing (ACVP_KTS_SCHEMA* or
 *	ACVP_KAS1_SCHEMA_* or ACVP_KAS2_SCHEMA_*)
 * @val key_generation_method [in] The private key generation method
 *	(ACVP_KAS_KEYGEN_*)
 * @val modulus [in] RSA modulus in use for key generation in bits
 * @val keylen [in] The length of key to derive / transport in bits
 * @val iut_id [in] The IUT's identifier - the data is held as an ASCII string
 *	that is to be used directly as is
 * @val server_id [in] The server's identifier - the data is held as an ASCII
 *	string that is to be used directly as is
 *
 * @val kts_hash [in] KTS configuration: The hash algorithm used for the OAEP
 *	function
 *
 * TODO kts_assoc_data_pattern WILL change in the future pending issue #800
 * @val kts_assoc_data_pattern [in] The pattern used for constructing the
 *	associated data - currently the string as documented in the KAS IFC
 *	specification section 3.4.1.4 is held here. This value may be NULL if
 *	no associated data is to be processed.
 *
 * @val kts_encoding [in] The encoding type used when constructing the
 * 	associated data (ACVP_KAS_ENCODING_*).
 *
 * @val mac [in] The MAC type used in key confirmation.
 * @val mac_keylen [in] The number of bits to take from the derived keying
 *	material to use for as the MAC key in key confirmation.
 * @val mac_maclen [in] The number of bits to use for the MAC tag.
 *
 * @val u [in] Union of data structures where the relevant data structure is
 *	filled as outlined above.
 */
struct kts_ifc_data {
	uint64_t schema;
	uint64_t key_generation_method;
	uint32_t modulus;
	uint32_t keylen;
	struct buffer iut_id;
	struct buffer server_id;

	/* KTS configuration */
	uint64_t kts_hash;
	struct buffer kts_assoc_data_pattern;
	uint64_t kts_encoding;

	/* MAC configuration */
	uint64_t mac;
	uint32_t mac_keylen;
	uint32_t mac_maclen;

	union {
		struct kts_ifc_init_data kts_ifc_init;
		struct kts_ifc_init_validation_data kts_ifc_init_validation;
		struct kts_ifc_resp_data kts_ifc_resp;
		struct kts_ifc_resp_validation_data kts_ifc_resp_validation;
	} u;
};

/**
 * @brief Callback data structure that must be implemented by the backend.
 *
 * All functions return 0 on success or != 0 on error.
 *
 * @val kts_ifc Perform KTS IFC testing
 */
struct kts_ifc_backend {
	int (*kts_ifc)(struct kts_ifc_data *data,  flags_t parsed_flags);
};

void register_kts_ifc_impl(struct kts_ifc_backend *implementation);

#ifdef __cplusplus
}
#endif

#endif /* PARSER_IFC_H */
