/*
 * Copyright (C) 2018 - 2020, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef _PARSER_DH_H
#define _PARSER_DH_H

#include "parser.h"
#include "parser_flags.h"

/**
 * @brief DH shashed shared secret generation
 *
 * @var cipher [in] ECC curve containing an OR of: one curve out of
 * 		      CURVEMASK, one hash out of HASHMASK, one MAC out
 * 		      of HMACMASK when using KDF.
 * @var P [in] Domain parameter P
 * @var Q [in] Domain parameter Q
 * @var G [in] Domain parameter G
 * @var Yrem [in] Remote public key
 * @var Xloc [disregard]
 * @var Yloc [out] Local public key generated by the IUT
 * @var hashzz [out] hashed shared secret
 */
struct dh_ss_data {
	uint64_t cipher;
	struct buffer P;
	struct buffer Q;
	struct buffer G;
	struct buffer Yrem;
	struct buffer Xloc;
	struct buffer Yloc;
	struct buffer hashzz;
};

/**
 * @brief DH shashed shared secret verification
 *
 * @var cipher [in] ECC curve containing an OR of: one curve out of
 * 		      CURVEMASK, one hash out of HASHMASK, one MAC out
 * 		      of HMACMASK when using KDF.
 * @var P [in] Domain parameter P
 * @var Q [in] Domain parameter Q
 * @var G [in] Domain parameter G
 * @var Yrem [in] Remote pubkey
 * @var Xloc [in] Private local key
 * @var Yloc [in] Local public key
 * @var hashzz [in] hashed shared secret
 * @var validity_success [out] Does the generated shared secret match with
 *				 @var hashzz (1) or whether it does not match
 *				 (0).
 */
struct dh_ss_ver_data {
	uint64_t cipher;
	struct buffer P;
	struct buffer Q;
	struct buffer G;
	struct buffer Yrem;
	struct buffer Xloc;
	struct buffer Yloc;
	struct buffer hashzz;
	uint32_t validity_success;
};

/**
 * @brief Callback data structure that must be implemented by the backend. Some
 *	  callbacks only need to be implemented if the respective cipher support
 *	  shall be tested.
 *
 * All functions return 0 on success or != 0 on error. Note, a failure in the
 * validity check @var dh_ss_ver due to a mismatch between the expected
 * and the actual shared secret is expected. In such cases, the validity test
 * error is still considered to be a successful operation and the return code
 * should be 0. Only if some general error is
 * detected a return code != must be returned.
 *
 * @var dh_ss DH shared secret generation
 * @var dh_ss_ver DH shared secret verification
 */
struct dh_backend {
	int (*dh_ss)(struct dh_ss_data *data, flags_t parsed_flags);
	int (*dh_ss_ver)(struct dh_ss_ver_data *data, flags_t parsed_flags);
};

void register_dh_impl(struct dh_backend *implementation);

#endif /* _PARSER_DH_H */
