/*
 * Non-physical true random number generator based on timing jitter.
 *
 * Copyright Stephan Mueller <smueller@chronox.de>, 2013
 *
 * License
 * =======
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU General Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#ifndef _JITTERENTROPY_H
#define _JITTERENTROPY_H

#ifdef __KERNEL__
#include "jitterentropy-base-kernel.h"
#else
#include "jitterentropy-base-user.h"
#endif /* __KERNEL__ */

/* Statistical data from the entropy source */
struct entropy_stat {
	unsigned int bitslot[64];	/* Counter for the bits set per bit
					   position in ->data */
	unsigned int bitvar[64];	/* Counter for the number of bit
					   variations per bit position in
					   ->data */
	unsigned int enable_bit_test;	/* enable bit test
       					   this flag is vital for the accuracy
					   of the statistic tests: when we
					   do the time measurements, we want
					   the observed entropy collection
					   loop executed as fast as the
					   unmeasured loop, i.e. without
					   the bit statistic logic; on the
					   other hand, the bit statistics
					   test is not interested in exact
					   timing */
	__u64 collection_begin;		/* timer for beginning of one
					   entropy collection round */
	__u64 collection_end;		/* timer for end of one round */
	__u64 old_delta;		/* Time delta of previous round to
					   calculate delta of deltas */
	unsigned int setbits;		/* see _jent_calc_statistic */
	unsigned int varbits;		/* see _jent_calc_statistic */
	unsigned int obsbits;		/* see _jent_calc_statistic */
	unsigned int collection_loop_set;	/* allow caller to set
						   collection_loop_cnt */
	unsigned int collection_loop_cnt;	/* Collection loop counter */
};

/* The entropy pool */
struct rand_data
{
	/* all data values that are vital to maintain the security
	 * of the RNG are marked as SENSITIVE. A user must not
	 * access that information while the RNG executes its loops to
	 * calculate the next random value. */
	__u64 data;		/* SENSITIVE Actual random number */
	__u64 prev_time;	/* SENSITIVE Previous time stamp */
#define DATA_SIZE_BITS ((sizeof(__u64)) * 8)
	__u64 old_data;		/* SENSITIVE FIPS continuous test */
	unsigned int fips_test:1;	/* FIPS integrity test enabled */
	unsigned int fips_fail:1;	/* FIPS status */
#ifdef CRYPTO_CPU_JITTERENTROPY_STAT
	struct entropy_stat entropy_stat;
#endif
};

/* Number of low bits of the time value that we want to consider */
#define TIME_ENTROPY_BITS 1

#define DRIVER_NAME     "jitterentropy"

/* -- BEGIN Main interface functions -- */

/* Number of low bits of the time value that we want to consider */
/* get raw entropy */
int jent_read_entropy(struct rand_data *entropy_collector,
		      char *data, size_t len);
/* intialize an instance of the entropy collector */
struct rand_data *jent_entropy_collector_alloc(void);
/* clearing of entropy collector */
void jent_entropy_collector_free(struct rand_data *entropy_collector);

/* initialization of entropy collector */
int jent_entropy_init(void);

/* -- END of Main interface functions -- */

/* -- BEGIN statistical test functions only complied with CRYPTO_CPU_JITTERENTROPY_STAT -- */

void _jent_init_statistic(struct rand_data *entropy_collector);
void _jent_calc_statistic(struct rand_data *entropy_collector,
			  struct entropy_stat *stat, unsigned int loop_cnt);
void _jent_bit_count(struct rand_data *entropy_collector, __u64 prev_data);

#ifdef CRYPTO_CPU_JITTERENTROPY_STAT
#define jent_init_statistic(x)    do { _jent_init_statistic(x); }    while (0)
#define jent_calc_statistic(x, y, z) do { _jent_calc_statistic(x, y, z); } while (0)
#define jent_bit_count(x,y)       do { _jent_bit_count(x, y); }      while (0)
void jent_gen_entropy_stat(struct rand_data *entropy_collector,
			   struct entropy_stat *stat);
void jent_fold_time_stat(__u64 *fold, unsigned int *loop_cnt);
#else /* CRYPTO_CPU_JITTERENTROPY_STAT */
#define jent_init_statistic(x)    do { if(0) _jent_init_statistic(x); }    while (0)
#define jent_calc_statistic(x, y, z) do { if(0) _jent_calc_statistic(x, y, z); } while (0)
#define jent_bit_count(x,y)       do { if(0) _jent_bit_count(x,y); }       while (0)
#endif /* CRYPTO_CPU_JITTERENTROPY_STAT */

/* -- END of statistical test function -- */

#endif /* _JITTERENTROPY_H */
